/******************************************************************************\
 * Technische Universitaet Darmstadt, Institut fuer Nachrichtentechnik
 * Copyright (c) 2001-2014
 *
 * Author(s):
 *  Volker Fischer, Julian Cable, Oliver Haffenden, Andrew Murphy
 *
 * Description:
  * Implements Digital Radio Mondiale (DRM) Multiplex Distribution Interface
 *  (MDI), Receiver Status and Control Interface (RSCI)
 *  and Distribution and Communications Protocol (DCP) as described in
 *  ETSI TS 102 820,  ETSI TS 102 349 and ETSI TS 102 821 respectively.
 *
 *  All modules that generate MDI information are given (normally at construction) a pointer to an MDI object.
 *  They call methods in this interface when they have MDI/RSCI data to impart.
 *
 *  Note that this previously needed QT, but now the QT socket is wrapped in an abstract class so
 *  this class can be compiled without QT. (A null socket is instantiated instead in this case, so
 *  nothing will actually happen.) This could be developed further by using a factory class to make
 *  the socket, in which case this class would only need to know about the abstract interface
 *  CPacketSocket.
 *
 *  This class is now almost a facade for all of the DCP and TAG classes, designed to have the
 *  same interface as the old CMDI class had. It could be improved further by moving the
 *  MDI generation into a separate class.
 *
 ******************************************************************************
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
\******************************************************************************/

#include "PacketSocket.h"
#include "../DrmReceiver.h"
#include "PacketSourceFile.h"
#include <sstream>
#include <iomanip>
#include "MDIRSCI.h"

/* Implementation *************************************************************/
CDownstreamDI::CDownstreamDI() : iLogFraCnt(0), pDrmReceiver(NULL),
    bMDIOutEnabled(false), bMDIInEnabled(false),bIsRecording(false),
    iFrequency(0), strRecordType(),
    vecTagItemGeneratorStr(MAX_NUM_STREAMS), vecTagItemGeneratorRBP(MAX_NUM_STREAMS),
    RSISubscribers(),pRSISubscriberFile(new CRSISubscriberFile)
{
    /* Initialise all the generators for strx and rbpx tags */
    for (int i=0; i<MAX_NUM_STREAMS; i++)
    {
        vecTagItemGeneratorStr[i].SetStreamNumber(i);
        vecTagItemGeneratorRBP[i].SetStreamNumber(i);
    }

    /* Reset all tags for initialization */
    ResetTags();

    /* Init constant tag */
    TagItemGeneratorProTyMDI.GenTag();
    TagItemGeneratorProTyRSCI.GenTag();

    /* Add the file subscriber to the list of subscribers */
    RSISubscribers.push_back(pRSISubscriberFile);

}

CDownstreamDI::~CDownstreamDI()
{
    for(vector<CRSISubscriber*>::iterator i = RSISubscribers.begin();
            i!=RSISubscribers.end(); i++)
    {
        delete *i;
    }
}

/******************************************************************************\
* DI send status, receive control                                             *
\******************************************************************************/
/* Access functions ***********************************************************/
void CDownstreamDI::SendLockedFrame(CParameter& Parameter,
                                    CSingleBuffer<_BINARY>& FACData,
                                    CSingleBuffer<_BINARY>& SDCData,
                                    vector<CSingleBuffer<_BINARY> >& vecMSCData
                                   )
{
    TagItemGeneratorFAC.GenTag(Parameter, FACData);
    TagItemGeneratorSDC.GenTag(Parameter, SDCData);
    //for (size_t i = 0; i < vecMSCData.size(); i++)
    for (size_t i = 0; i < MAX_NUM_STREAMS; i++)
    {
        vecTagItemGeneratorStr[i].GenTag(Parameter, vecMSCData[i]);
    }
    TagItemGeneratorRobMod.GenTag(Parameter.GetWaveMode());
    TagItemGeneratorRxDemodMode.GenTag(Parameter.GetReceiverMode(), Parameter.GetWaveMode());

    /* SDC channel information tag must be created here because it must be sent
       with each AF packet */
    TagItemGeneratorSDCChanInf.GenTag(Parameter);

    TagItemGeneratorRINF.GenTag(Parameter.sReceiverID); /* rinf */

    /* RSCI tags ------------------------------------------------------------ */
    TagItemGeneratorRAFS.GenTag(Parameter);
    TagItemGeneratorRWMF.GenTag(true, Parameter.rWMERFAC); /* WMER for FAC */
    TagItemGeneratorRWMM.GenTag(true, Parameter.rWMERMSC); /* WMER for MSC */
    TagItemGeneratorRMER.GenTag(true, Parameter.rMER); /* MER for MSC */
    TagItemGeneratorRDEL.GenTag(true, Parameter.vecrRdelThresholds, Parameter.vecrRdelIntervals);
    TagItemGeneratorRDOP.GenTag(true, Parameter.rRdop);
    TagItemGeneratorRINT.GenTag(true,Parameter.rIntFreq, Parameter.rINR, Parameter.rICR);
    TagItemGeneratorRNIP.GenTag(true,Parameter.rMaxPSDFreq, Parameter.rMaxPSDwrtSig);
    TagItemGeneratorRxService.GenTag(true, Parameter.GetCurSelAudioService());
    TagItemGeneratorReceiverStatus.GenTag(Parameter);
    TagItemGeneratorRxFrequency.GenTag(true, Parameter.GetFrequency()); /* rfre */
    TagItemGeneratorRxActivated.GenTag(true); /* ract */
    TagItemGeneratorPowerSpectralDensity.GenTag(Parameter);
    TagItemGeneratorPowerImpulseResponse.GenTag(Parameter);
    TagItemGeneratorPilots.GenTag(Parameter);

    /* Generate some other tags */
    _REAL rSigStr = Parameter.SigStrstat.getCurrent();
    TagItemGeneratorSignalStrength.GenTag(true, rSigStr + S9_DBUV);

    TagItemGeneratorGPS.GenTag(true, Parameter.gps_data);   // rgps

    GenDIPacket();
}

void CDownstreamDI::SendUnlockedFrame(CParameter& Parameter)
{
    /* This is called once per frame if the receiver is unlocked */

    /* In the MDI profile, we used to ignore this altogether since "I assume there's no point */
    /* in generating empty packets with no reception monitoring information" */
    /* But now there could be multiple profiles at the same time. TODO: decide what to do! */
    /*  if (cProfile == 'M')
            return;*/

    /* Send empty tags for most tag items */
    ResetTags();

    TagItemGeneratorFAC.GenEmptyTag();
    TagItemGeneratorSDC.GenEmptyTag();

    /* mode is unknown - make empty robm tag */
    TagItemGeneratorRobMod.GenEmptyTag();

    TagItemGeneratorRxDemodMode.GenTag(Parameter.GetReceiverMode(), Parameter.GetWaveMode());

    TagItemGeneratorSDCChanInf.GenEmptyTag();

    TagItemGeneratorReceiverStatus.GenTag(Parameter);

    TagItemGeneratorPowerSpectralDensity.GenTag(Parameter);

    TagItemGeneratorPowerImpulseResponse.GenEmptyTag();

    TagItemGeneratorPilots.GenEmptyTag();

    TagItemGeneratorRNIP.GenTag(true,Parameter.rMaxPSDFreq, Parameter.rMaxPSDwrtSig);

    /* Generate some other tags */
    TagItemGeneratorRINF.GenTag(Parameter.sReceiverID); /* rinf */
    TagItemGeneratorRxFrequency.GenTag(true, Parameter.GetFrequency()); /* rfre */
    TagItemGeneratorRxActivated.GenTag(true); /* ract */
    _REAL rSigStr = Parameter.SigStrstat.getCurrent();
    TagItemGeneratorSignalStrength.GenTag(true, rSigStr + S9_DBUV);

    TagItemGeneratorGPS.GenTag(true, Parameter.gps_data);   /* rgps */

    GenDIPacket();
}

void CDownstreamDI::SendAMFrame(CParameter& Parameter, CSingleBuffer<_BINARY>& CodedAudioData)
{
    /* This is called once per 400ms if the receiver is in AM mode */

    /* In the MDI profile, ignore this altogether since there's no DRM information */
    /*if (cProfile == 'M')
        return;*/

    /* Send empty tags for most tag items */
    ResetTags();

    TagItemGeneratorFAC.GenEmptyTag();
    TagItemGeneratorSDC.GenEmptyTag();
    /* mode is unknown - make empty robm tag */
    TagItemGeneratorRobMod.GenEmptyTag();

    /* demod mode */
    TagItemGeneratorRxDemodMode.GenTag(Parameter.GetReceiverMode(), Parameter.GetWaveMode());

    TagItemGeneratorSDCChanInf.GenEmptyTag();

    /* These will be set appropriately when the rx is put into AM mode */
    /* We need to decide what "appropriate" settings are */
    TagItemGeneratorReceiverStatus.GenTag(Parameter);

    TagItemGeneratorPowerSpectralDensity.GenTag(Parameter);

    TagItemGeneratorPowerImpulseResponse.GenEmptyTag();

    TagItemGeneratorPilots.GenEmptyTag();

    TagItemGeneratorRNIP.GenTag(true, Parameter.rMaxPSDFreq, Parameter.rMaxPSDwrtSig);

    // Generate a rama tag with the encoded audio data
    TagItemGeneratorAMAudio.GenTag(Parameter, CodedAudioData);

    /* Generate some other tags */
    TagItemGeneratorRINF.GenTag(Parameter.sReceiverID); /* rinf */
    TagItemGeneratorRxFrequency.GenTag(true, Parameter.GetFrequency()); /* rfre */
    TagItemGeneratorRxActivated.GenTag(true); /* ract */
    _REAL rSigStr = Parameter.SigStrstat.getCurrent();
    TagItemGeneratorSignalStrength.GenTag(true, rSigStr + S9_DBUV);

    TagItemGeneratorGPS.GenTag(true, Parameter.gps_data);   /* rgps */

    GenDIPacket();
}

void CDownstreamDI::SetReceiver(CDRMReceiver *pReceiver)
{
    pDrmReceiver = pReceiver;
    for(vector<CRSISubscriber*>::iterator i = RSISubscribers.begin();
            i!=RSISubscribers.end(); i++)
        (*i)->SetReceiver(pReceiver);
}

/* Actual DRM DI protocol implementation *****************************************/
void CDownstreamDI::GenDIPacket()
{
    /* Reset the tag packet generator */
    TagPacketGenerator.Reset();

    /* Increment MDI packet counter and generate counter tag */
    TagItemGeneratorLoFrCnt.GenTag();

    /* dlfc tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorLoFrCnt);

    /* *ptr tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorProTyMDI);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorProTyRSCI);

    /* rinf taf */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRINF);

    /* rgps tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorGPS);

    /* rpro tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorProfile);

    /* rdmo - note that this is currently empty */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRxDemodMode);

    /* ract */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRxActivated);

    /* rfre tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRxFrequency);

    /* fac_ tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorFAC);

    /* sdc_ tag - don't send if empty */
    if(TagItemGeneratorSDC.GetTotalLength()>64)
        TagPacketGenerator.AddTagItem(&TagItemGeneratorSDC);

    /* sdci tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorSDCChanInf);

    /* robm tag */
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRobMod);

    /* strx tag */
    size_t i;
    for (i = 0; i < MAX_NUM_STREAMS; i++)
    {
        TagPacketGenerator.AddTagItem(&vecTagItemGeneratorStr[i]);
    }
    TagPacketGenerator.AddTagItem(&TagItemGeneratorAMAudio);

    TagPacketGenerator.AddTagItem(&TagItemGeneratorSignalStrength);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRAFS);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRMER);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRWMM);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRWMF);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRDEL);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRDOP);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRINT);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorRNIP);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorReceiverStatus);

    TagPacketGenerator.AddTagItem(&TagItemGeneratorPowerSpectralDensity);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorPowerImpulseResponse);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorPilots);


    for (i = 0; i < MAX_NUM_STREAMS; i++)
    {
        TagPacketGenerator.AddTagItem(&vecTagItemGeneratorRBP[i]);
    }

    /*return TagPacketGenerator.GenAFPacket(bUseAFCRC);*/

    /* transmit a packet to each subscriber */
    for(vector<CRSISubscriber*>::iterator s = RSISubscribers.begin();
            s!=RSISubscribers.end(); s++)
    {
        // re-generate the profile tag for each subscriber
        TagItemGeneratorProfile.GenTag((*s)->GetProfile());
        (*s)->TransmitPacket(TagPacketGenerator);
    }
}

void CDownstreamDI::ResetTags()
{
    /* Do not reset "*ptr" tag because this one is static */
    /* This group of tags are generated each time so don't need empty tags generated */
    TagItemGeneratorLoFrCnt.Reset(); /* dlfc tag */
    TagItemGeneratorFAC.Reset(); /* fac_ tag */
    TagItemGeneratorSDCChanInf.Reset(); /* sdci tag */
    TagItemGeneratorRobMod.Reset(); /* robm tag */
    TagItemGeneratorRINF.Reset(); /* info tag */
    TagItemGeneratorReceiverStatus.Reset(); /* rsta */

    TagItemGeneratorProfile.Reset(); /* rpro */
    TagItemGeneratorGPS.Reset();    /* rgps */

    TagItemGeneratorPowerSpectralDensity.Reset();
    TagItemGeneratorPowerImpulseResponse.Reset();
    TagItemGeneratorPilots.Reset();

    /* This group of tags might not be generated, so make an empty version in case */

    TagItemGeneratorSignalStrength.GenEmptyTag(); /* rdbv tag */
    TagItemGeneratorRWMF.GenEmptyTag(); /* rwmf tag */
    TagItemGeneratorRWMM.GenEmptyTag(); /* rwmm tag */
    TagItemGeneratorRMER.GenEmptyTag(); /* rmer tag */
    TagItemGeneratorRDEL.GenEmptyTag(); /* rdel tag */
    TagItemGeneratorRDOP.GenEmptyTag(); /* rdop tag */
    TagItemGeneratorRINT.GenEmptyTag(); /* rint tag */
    TagItemGeneratorRNIP.GenEmptyTag(); /* rnip tag */
    TagItemGeneratorRAFS.GenEmptyTag(); /* rafs tag */

    /* Tags that are not fully implemented yet */
    TagItemGeneratorRxBandwidth.GenEmptyTag(); /* rbw_ */
    TagItemGeneratorRxService.GenEmptyTag(); /* rser */

    for (int i = 0; i < MAX_NUM_STREAMS; i++)
    {
        vecTagItemGeneratorStr[i].Reset(); // strx tag
        vecTagItemGeneratorRBP[i].GenEmptyTag(); // make empty version of mandatory tag that isn't implemented
    }

    TagItemGeneratorAMAudio.Reset(); // don't make the tag in DRM mode

    TagItemGeneratorSDC.GenEmptyTag();
}

void CDownstreamDI::GetNextPacket(CSingleBuffer<_BINARY>&)
{
    // TODO
}

bool
CDownstreamDI::AddSubscriber(const string& dest, const char profile, const string& origin)
{
    CRSISubscriber* subs = NULL;
    /* heuristic to test for file or socket - TODO - better syntax */
    size_t p = dest.find_first_not_of("TPtp0123456789.:");
    if (p != string::npos)
    {
        subs = new CRSISubscriberFile();
    }
    else
    {
        subs = new CRSISubscriberSocket(NULL);
    }

    if(subs == NULL)
    {
        cerr << "can't make RSI Subscriber" << endl;
        return false;
    }

    // Delegate
    bool bOK = true;
    if (dest != "")
    {
        bOK &= subs->SetDestination(dest);
        if (bOK)
        {
            bMDIOutEnabled = true;
            subs->SetProfile(profile);
        }
    }
    if (origin != "")
    {
        bOK &= subs->SetOrigin(origin);
        if (bOK)
        {
            bMDIInEnabled = true;
        }
    }
    if (bOK)
    {
        subs->SetReceiver(pDrmReceiver);
        RSISubscribers.push_back(subs);
        return true;
    }
    else
    {
        bMDIInEnabled = false;
        bMDIOutEnabled = false;
        delete subs;
    }
    return false;
}

bool CDownstreamDI::SetOrigin(const string&)
{
    return false;
}

bool CDownstreamDI::SetDestination(const string&)
{
    return false;
}

bool CDownstreamDI::GetDestination(string&)
{
    return false; // makes no sense
}

void CDownstreamDI::SetAFPktCRC(const bool bNAFPktCRC)
{
    for(vector<CRSISubscriber*>::iterator i = RSISubscribers.begin();
            i!=RSISubscribers.end(); i++)
        (*i)->SetAFPktCRC(bNAFPktCRC);
}

string CDownstreamDI::GetRSIfilename(CParameter& Parameter, const char cProfile)
{
    /* Get current UTC time */
    time_t ltime;
    time(&ltime);
    struct tm* gmtCur = gmtime(&ltime);

    iFrequency = Parameter.GetFrequency(); // remember this for later

    stringstream filename;
    filename << Parameter.sReceiverID << "_";
    filename << setw(4) << setfill('0') << gmtCur->tm_year + 1900 << "-" << setw(2) << setfill('0')<< gmtCur->tm_mon + 1;
    filename << "-" << setw(2) << setfill('0')<< gmtCur->tm_mday << "_";
    filename << setw(2) << setfill('0') << gmtCur->tm_hour << "-" << setw(2) << setfill('0')<< gmtCur->tm_min;
    filename << "-" << setw(2) << setfill('0')<< gmtCur->tm_sec << "_";
    filename << setw(8) << setfill('0') << (iFrequency*1000) << ".rs" << char(toupper(cProfile));
    return filename.str();
}

void CDownstreamDI::SetRSIRecording(CParameter& Parameter, const bool bOn, char cPro, const string& type)
{
    strRecordType = type;
    if (bOn)
    {
        pRSISubscriberFile->SetProfile(cPro);
        string fn = GetRSIfilename(Parameter, cPro);
        if(strRecordType != "" && strRecordType != "raw")
            fn += "." + strRecordType;

        pRSISubscriberFile->SetDestination(fn);
        pRSISubscriberFile->StartRecording();
        bMDIOutEnabled = true;
        bIsRecording = true;
    }
    else
    {
        pRSISubscriberFile->StopRecording();
        bIsRecording = false;
    }
}

/* needs to be called in case a new RSCI file needs to be started */
void CDownstreamDI::NewFrequency(CParameter& Parameter)
{
    /* Has it really changed? */
    int iNewFrequency = Parameter.GetFrequency();

    if (iNewFrequency != iFrequency)
    {
        if (bIsRecording)
        {
            pRSISubscriberFile->StopRecording();
            string fn = GetRSIfilename(Parameter, pRSISubscriberFile->GetProfile());
            if(strRecordType != "" && strRecordType != "raw")
                fn += "." + strRecordType;
            pRSISubscriberFile->SetDestination(fn);
            pRSISubscriberFile->StartRecording();
        }
    }
}

void CDownstreamDI::SendPacket(const vector<_BYTE>&, uint32_t, uint16_t)
{
    cerr << "this shouldn't get called CDownstreamDI::SendPacket" << endl;
}

void CDownstreamDI::poll()
{
    for(vector<CRSISubscriber*>::iterator i = RSISubscribers.begin();
            i!=RSISubscribers.end(); i++) {
        CRSISubscriber *s = *i;
        string origin;
        bool hasOrigin = s->GetOrigin(origin);
        if(hasOrigin)
            s->poll();
    }
}

/* allow multiple destinations, allow destinations to send cpro instructions back */
/******************************************************************************\
* DI receive status, send control                                             *
\******************************************************************************/
CUpstreamDI::CUpstreamDI() : source(NULL), sink(), bUseAFCRC(true), bMDIOutEnabled(false), bMDIInEnabled(false)
{
    /* Init constant tag */
    TagItemGeneratorProTyRSCI.GenTag();
}

CUpstreamDI::~CUpstreamDI()
{
    if(source)
    {
        delete source;
    }
}

bool CUpstreamDI::SetOrigin(const string& str)
{
    /* only allow one listening address */
    if(bMDIInEnabled)
        return false;

    if(source)
        return false;

    strOrigin = str;

    // try a file
    source = new CPacketSourceFile;
    bool bOK = source->SetOrigin(str);

    if(!bOK)
    {
        // try a socket
        delete source;
        source = new CPacketSocketNative;
        bOK = source->SetOrigin(str);
    }
    if (bOK)
    {
        source->SetPacketSink(this);
        bMDIInEnabled = true;
        return true;
    }
    return false;
}

bool CUpstreamDI::SetDestination(const string& str)
{

    bMDIOutEnabled = sink.SetDestination(str);

    return bMDIOutEnabled;
}

bool CUpstreamDI::GetDestination(string& str)
{
    return sink.GetDestination(str);
}

void CUpstreamDI::SetFrequency(int iNewFreqkHz)
{
    if(bMDIOutEnabled==false)
        return;
    TagPacketGenerator.Reset();
    TagItemGeneratorCfre.GenTag(iNewFreqkHz);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorProTyRSCI);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorCfre);
    sink.TransmitPacket(TagPacketGenerator);
}

void CUpstreamDI::SetReceiverMode(ERecMode eNewMode)
{
    if(bMDIOutEnabled==false)
        return;
    TagPacketGenerator.Reset();
    TagItemGeneratorCdmo.GenTag(eNewMode);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorProTyRSCI);
    TagPacketGenerator.AddTagItem(&TagItemGeneratorCdmo);
    sink.TransmitPacket(TagPacketGenerator);
}

/* we only support one upstream RSCI source, so ignore the source address */
void CUpstreamDI::SendPacket(const vector<_BYTE>& vecbydata, uint32_t, uint16_t)
{
    if(!vecbydata.size())
        return;
    if(vecbydata[0]=='P')
    {
        vector<_BYTE> vecOut;
        if(Pft.DecodePFTPacket(vecbydata, vecOut))
        {
            queue.Put(vecOut);
        }
    }
    else
        queue.Put(vecbydata);
}

void CUpstreamDI::InitInternal(CParameter&)
{
    iInputBlockSize = 1; /* anything is enough but not zero */
    iMaxOutputBlockSize = 2048*SIZEOF__BYTE; /* bigger than an ethernet packet */
}

void CUpstreamDI::ProcessDataInternal(CParameter&)
{
    vector<_BYTE> vecbydata;
    source->poll();
    queue.Get(vecbydata);
    size_t bytes = vecbydata.size();
    iOutputBlockSize = bytes*SIZEOF__BYTE;
    pvecOutputData->Init(iOutputBlockSize);
    pvecOutputData->ResetBitAccess();
    for(size_t i=0; i<bytes; i++)
    {
        pvecOutputData->Enqueue(vecbydata[i], SIZEOF__BYTE);
    }
}
