#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "SDL/SDL.h" 
#include "SDL/SDL_opengl.h"

#include "color.h"

/* User for parser debug */
#define PARSER_UNIT_TEST 0

#define SCREEN_WIDTH 600
#define SCREEN_HEIGHT 400 
// #define SCREEN_WIDTH 640
// #define SCREEN_HEIGHT 480 
#define BORDER_SIZE 5

/* FCG stuff */
float reflectance[24][36];
char color_names[24][64];

int lambda_min = 0;
int lambda_max = 0;
int lambda_count = 0;
int lambda_step = 0;
int color_count = 0;


/* SDL and OpenGL stuff */
/* Information about the current video settings. */
const SDL_VideoInfo* info = NULL;
//Event handler
SDL_Event event;
/* Dimensions of our window. */
int width = 0;
int height = 0;
/* Color depth in bits of our window. */
int bpp = 0;
/* Flags we will pass into SDL_SetVideoMode. */
int flags = 0;


int graphics_system_start() 
{
    /* First, initialize SDL's video subsystem. */
    if(SDL_Init( SDL_INIT_VIDEO ) < 0) {
        /* Failed, exit. */
        fprintf(stderr, "Video initialization failed: %s\n", SDL_GetError());
        return -1;
    }
    
    /* Let's get some video information. */
    info = SDL_GetVideoInfo();
    if(!info) {
        /* This should probably never happen. */
        fprintf(stderr, "Video query failed: %s\n", SDL_GetError());
        return -1;
    }
    
    /* our resolution */
    width = SCREEN_WIDTH;
    height = SCREEN_HEIGHT;
    bpp = info->vfmt->BitsPerPixel;

    /*
     * Now, we want to setup our requested
     * window attributes for our OpenGL window.
     * We want *at least* 5 bits of red, green
     * and blue. We also want at least a 16-bit
     * depth buffer.
     *
     * The last thing we do is request a double
     * buffered window. '1' turns on double
     * buffering, '0' turns it off.
     *
     * Note that we do not use SDL_DOUBLEBUF in
     * the flags to SDL_SetVideoMode. That does
     * not affect the GL attribute state, only
     * the standard 2D blitting setup.
     */
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 24 );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

    flags = SDL_OPENGL;

    /*
     * Set the video mode
     */
    if(SDL_SetVideoMode(width, height, bpp, flags) == 0) 
    {
        /* 
         * This could happen for a variety of reasons,
         * including DISPLAY not being set, the specified
         * resolution not being available, etc.
         */
        fprintf(stderr, "Video mode set failed: %s\n", SDL_GetError());
        return -1;
    }

    SDL_WM_SetCaption("OpenGL FCG T0 (M. GATTASS)", NULL);

    //Initialize Projection Matrix 
    glMatrixMode( GL_PROJECTION ); 
    glLoadIdentity(); 
    glOrtho( 0.0, SCREEN_WIDTH, SCREEN_HEIGHT, 0.0, 1.0, -1.0 );
    glTranslatef( SCREEN_WIDTH / 2.f, SCREEN_HEIGHT / 2.f, 0.f );
    glClear(GL_COLOR_BUFFER_BIT); 

    return 0;

}

void graphics_system_stop()
{
    SDL_Quit();
}

void graphics_system_draw()
{
    SDL_GL_SwapBuffers();
}

void graphics_main_loop()
{
    int quit = 0;
    
    while (quit == 0) 
    {
	while(SDL_PollEvent(&event))
	{
	    if(event.type == SDL_QUIT)
	    {
                quit = 1;
            }
	}
    }
}

void FillRect(float r, float g, float b) 
{
    static int x = 0;
    static int y = 0;

    glBegin(GL_QUADS); 

    glColor3f( r, g, b );

    glVertex2f( ((100*x) - (SCREEN_WIDTH/2)) + BORDER_SIZE, ((100*y) - (SCREEN_HEIGHT/2)) + BORDER_SIZE); 
    x++;
    glVertex2f( ((100*x) - (SCREEN_WIDTH/2)) - BORDER_SIZE, ((100*y) - (SCREEN_HEIGHT/2)) + BORDER_SIZE); 
    y++;
    glVertex2f( ((100*x) - (SCREEN_WIDTH/2)) - BORDER_SIZE, ((100*y) - (SCREEN_HEIGHT/2)) - BORDER_SIZE); 
    x--;
    glVertex2f( ((100*x) - (SCREEN_WIDTH/2)) + BORDER_SIZE, ((100*y) - (SCREEN_HEIGHT/2)) - BORDER_SIZE); 
    y--;

    glEnd(); 

    x++;
    if (x == 6) 
    {
	x = 0;
	y++;
    }

}

int parse_color_chart(char *input_filename) 
{
    FILE *fp;
    char temp1[64];
    int temp2;
    int color_index = 0;

    memset(temp1, 0, 64);

    // ad-hoc parser
    fp = fopen(input_filename, "r");

    if (fp == NULL) 
    {
	return -1;
    }

    // No.
    fscanf(fp, "%s", temp1);

    // Color name
    fscanf(fp, "%s", temp1);
    
    // first lambda
    fscanf(fp, "%d", &lambda_min);
    
    // second color, we assume the step is a constant
    fscanf(fp, "%d", &temp2);
    lambda_step = temp2 - lambda_min;
    
    temp2 = 0;
    do
    {
	lambda_max = temp2;
	fscanf(fp, "%d", &temp2);
    } while (temp2 != 1);
    
    // number of lambdas
    lambda_count = ((lambda_max - lambda_min) / lambda_step) + 1;
	
    do
    {
	fscanf(fp, "%s", color_names[color_index]);

	for (int i = 0; i < lambda_count; i++)
	{
	    fscanf(fp, "%f", &reflectance[color_index][i]);
	}
	color_index++;
    } while (fscanf(fp, "%d", &temp2) != EOF);

    color_count = color_index;

    /* parser unit test */
#if PARSER_UNIT_TEST

    fprintf(stderr, "lambda_count = %d\n\n", lambda_count);
    for (int i = 0; i < color_count; i++)
    {
	fprintf(stderr, "%s", color_names[i]);
	for (int j = 0; j < lambda_count; j++)
	{
	    fprintf(stderr, " %f", reflectance[i][j]); 
	}
	fprintf(stderr, "\n");		
    }
#endif

    fclose(fp);

    return 0;
    
}

int main(int argc, char *argv[]){
    float X = 0, Y = 0, Z = 0;
    float pR, pG, pB;
	
    if (argc != 2) {
	fprintf(stderr, "Usage:\nfcg_t0 macbeth.txt\n\n");			
	return EXIT_FAILURE;
    }
    
    if (parse_color_chart(argv[1]) < 0)
    {
	fprintf(stderr, "File %s could not be opened.\n", argv[1]);
	return EXIT_FAILURE;
    }

    if (graphics_system_start() < 0)
    {
	fprintf(stderr, "Error in the graphics subsystem.\n");
	return EXIT_FAILURE;
    }
    
    for (int i = 0; i < color_count; i++)
    {
	corCIEXYZfromSurfaceReflectance( (float) lambda_min, lambda_count, lambda_step, reflectance[i], &X, &Y, &Z, D65);
	
	fprintf(stderr, "\n%s\nX: %f Y: %f Z: %f\n", color_names[i], X, Y, Z);
	
	corCIEXYZtosRGB(X, Y, Z, &pR, &pG, &pB, D65);
	
	// normalize negative values to zero
	pR = (pR < 0.0)? 0.0 : pR;
	pG = (pG < 0.0)? 0.0 : pG;
	pB = (pB < 0.0)? 0.0 : pB;
	fprintf(stderr, "R: %f G: %f B: %f\n", pR, pG, pB);

	FillRect(pR, pG, pB);
    }

    graphics_system_draw();

    graphics_main_loop();
    
    graphics_system_stop();

    return EXIT_SUCCESS;
}
